/*
 * Decompiled with CFR 0.152.
 */
package com.sansec.ocsp.test;

import com.sansec.asn1.ASN1Object;
import com.sansec.asn1.ASN1OctetString;
import com.sansec.asn1.DERObjectIdentifier;
import com.sansec.asn1.DEROctetString;
import com.sansec.asn1.ocsp.OCSPObjectIdentifiers;
import com.sansec.asn1.x509.GeneralName;
import com.sansec.asn1.x509.X509Extension;
import com.sansec.asn1.x509.X509Extensions;
import com.sansec.asn1.x509.X509Name;
import com.sansec.jce.X509Principal;
import com.sansec.jce.provider.SwxaProvider;
import com.sansec.ocsp.BasicOCSPResp;
import com.sansec.ocsp.BasicOCSPRespGenerator;
import com.sansec.ocsp.CertificateID;
import com.sansec.ocsp.CertificateStatus;
import com.sansec.ocsp.OCSPReq;
import com.sansec.ocsp.OCSPReqGenerator;
import com.sansec.ocsp.OCSPResp;
import com.sansec.ocsp.OCSPRespGenerator;
import com.sansec.ocsp.Req;
import com.sansec.ocsp.SingleResp;
import com.sansec.ocsp.test.OCSPTestUtil;
import com.sansec.util.encoders.Base64;
import com.sansec.util.test.SimpleTest;
import com.sansec.x509.extension.X509ExtensionUtil;
import java.io.ByteArrayInputStream;
import java.math.BigInteger;
import java.security.KeyPair;
import java.security.Security;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.Random;
import java.util.Set;
import java.util.Vector;

public class OCSPTest
extends SimpleTest {
    byte[] testResp1 = Base64.decode("MIIFnAoBAKCCBZUwggWRBgkrBgEFBQcwAQEEggWCMIIFfjCCARehgZ8wgZwxCzAJBgNVBAYTAklOMRcwFQYDVQQIEw5BbmRocmEgcHJhZGVzaDESMBAGA1UEBxMJSHlkZXJhYmFkMQwwCgYDVQQKEwNUQ1MxDDAKBgNVBAsTA0FUQzEeMBwGA1UEAxMVVENTLUNBIE9DU1AgUmVzcG9uZGVyMSQwIgYJKoZIhvcNAQkBFhVvY3NwQHRjcy1jYS50Y3MuY28uaW4YDzIwMDMwNDAyMTIzNDU4WjBiMGAwOjAJBgUrDgMCGgUABBRs07IuoCWNmcEl1oHwIak1BPnX8QQUtGyl/iL9WJ1VxjxFj0hAwJ/s1AcCAQKhERgPMjAwMjA4MjkwNzA5MjZaGA8yMDAzMDQwMjEyMzQ1OFowDQYJKoZIhvcNAQEFBQADgYEAfbN0TCRFKdhsmvOdUoiJ+qvygGBzDxD/VWhXYA+16AphHLIWNABR3CgHB3zWtdy2j7DJmQ/R7qKj7dUhWLSqclAiPgFtQQ1YvSJAYfEIdyHkxv4NP0LSogxrumANcDyC9yt/W9yHjD2ICPBIqCsZLuLkOHYi5DlwWe9Zm9VFwCGgggPMMIIDyDCCA8QwggKsoAMCAQICAQYwDQYJKoZIhvcNAQEFBQAwgZQxFDASBgNVBAMTC1RDUy1DQSBPQ1NQMSYwJAYJKoZIhvcNAQkBFhd0Y3MtY2FAdGNzLWNhLnRjcy5jby5pbjEMMAoGA1UEChMDVENTMQwwCgYDVQQLEwNBVEMxEjAQBgNVBAcTCUh5ZGVyYWJhZDEXMBUGA1UECBMOQW5kaHJhIHByYWRlc2gxCzAJBgNVBAYTAklOMB4XDTAyMDgyOTA3MTE0M1oXDTAzMDgyOTA3MTE0M1owgZwxCzAJBgNVBAYTAklOMRcwFQYDVQQIEw5BbmRocmEgcHJhZGVzaDESMBAGA1UEBxMJSHlkZXJhYmFkMQwwCgYDVQQKEwNUQ1MxDDAKBgNVBAsTA0FUQzEeMBwGA1UEAxMVVENTLUNBIE9DU1AgUmVzcG9uZGVyMSQwIgYJKoZIhvcNAQkBFhVvY3NwQHRjcy1jYS50Y3MuY28uaW4wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGBAM+XWW4caMRv46D7L6Bv8iwtKgmQu0SAybmFRJiz12qXzdvTLt8C75OdgmUomxp0+gW/4XlTPUqOMQWv463aZRv9Ust4f8MHEJh4ekP/NS9+d8vEO3P40ntQkmSMcFmtA9E1koUtQ3MSJlcs441JjbgUaVnmjDmmniQnZY4bU3tVAgMBAAGjgZowgZcwDAYDVR0TAQH/BAIwADALBgNVHQ8EBAMCB4AwEwYDVR0lBAwwCgYIKwYBBQUHAwkwNgYIKwYBBQUHAQEEKjAoMCYGCCsGAQUFBzABhhpodHRwOi8vMTcyLjE5LjQwLjExMDo3NzAwLzAtBgNVHR8EJjAkMCKgIKAehhxodHRwOi8vMTcyLjE5LjQwLjExMC9jcmwuY3JsMA0GCSqGSIb3DQEBBQUAA4IBAQB6FovM3B4VDDZ15o12gnADZsIk9fTAczLlcrmXLNN4PgmqgnwF0Ymj3bD5SavDOXxbA65AZJ7rBNAguLUo+xVkgxmoBH7R2sBxjTCcr07NEadxM3HQkt0aX5XYEl8eRoifwqYAI9h0ziZfTNes8elNfb3DoPPjqq6VmMg0f0iMS4W8LjNPorjRB+kIosa1deAGPhq0eJ8yr0/s2QR2/WFD5P4aXc8IKWleklnIImS3zqiPrq6tl2Bm8DZj7vXlTOwmraSQxUwzCKwYob1yGvNOUQTqpG6jxn7jgDawHU1+WjWQe4Q34/pWeGLysxTraMa+Ug9kPe+jy/qRX2xwvKBZ====");
    byte[] testResp2 = Base64.decode("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==");
    private static byte[] irregReq = Base64.decode("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");

    public String getName() {
        return "OCSP";
    }

    private void testECDSA() throws Exception {
        byte[] extValue;
        ASN1Object extObj;
        Set extOids;
        byte[] reqEnc;
        OCSPReq newReq;
        Req[] requests;
        X509Certificate[] certs;
        String signDN = "O=Bouncy Castle, C=AU";
        KeyPair signKP = OCSPTestUtil.makeECKeyPair();
        X509Certificate testCert = OCSPTestUtil.makeECDSACertificate(signKP, signDN, signKP, signDN);
        String origDN = "CN=Eric H. Echidna, E=eric@bouncycastle.org, O=Bouncy Castle, C=AU";
        GeneralName origName = new GeneralName(new X509Name(origDN));
        CertificateID id = new CertificateID("1.3.14.3.2.26", testCert, BigInteger.valueOf(1L));
        OCSPReqGenerator gen = new OCSPReqGenerator();
        gen.addRequest(id);
        OCSPReq req = gen.generate();
        if (req.isSigned()) {
            this.fail("signed but shouldn't be");
        }
        if ((certs = req.getCerts("BC")) != null) {
            this.fail("null certs expected, but not found");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        X509Certificate[] chain = new X509Certificate[1];
        gen = new OCSPReqGenerator();
        gen.setRequestorName(new GeneralName(4, new X509Principal("CN=fred")));
        gen.addRequest(new CertificateID("1.3.14.3.2.26", testCert, BigInteger.valueOf(1L)));
        chain[0] = testCert;
        req = gen.generate("SHA1withECDSA", signKP.getPrivate(), chain, "BC");
        if (!req.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!req.verify(signKP.getPublic(), "BC")) {
            this.fail("signature failed to verify");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        if ((certs = req.getCerts("BC")) == null) {
            this.fail("null certs found");
        }
        if (certs.length != 1 || !certs[0].equals(testCert)) {
            this.fail("incorrect certs found in request");
        }
        if (!(newReq = new OCSPReq(reqEnc = req.getEncoded())).verify(signKP.getPublic(), "BC")) {
            this.fail("newReq signature failed to verify");
        }
        chain = new X509Certificate[1];
        gen = new OCSPReqGenerator();
        Vector<DERObjectIdentifier> oids = new Vector<DERObjectIdentifier>();
        Vector<X509Extension> values = new Vector<X509Extension>();
        byte[] sampleNonce = new byte[16];
        Random rand = new Random();
        rand.nextBytes(sampleNonce);
        gen.setRequestorName(new GeneralName(4, new X509Principal("CN=fred")));
        oids.addElement(OCSPObjectIdentifiers.id_pkix_ocsp_nonce);
        values.addElement(new X509Extension(false, (ASN1OctetString)new DEROctetString(new DEROctetString(sampleNonce))));
        gen.setRequestExtensions(new X509Extensions(oids, values));
        gen.addRequest(new CertificateID("1.3.14.3.2.26", testCert, BigInteger.valueOf(1L)));
        chain[0] = testCert;
        req = gen.generate("SHA1withECDSA", signKP.getPrivate(), chain, "BC");
        if (!req.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!req.verify(signKP.getPublic(), "BC")) {
            this.fail("signature failed to verify");
        }
        if ((extOids = req.getCriticalExtensionOIDs()).size() != 0) {
            this.fail("wrong number of critical extensions in OCSP request.");
        }
        if ((extOids = req.getNonCriticalExtensionOIDs()).size() != 1) {
            this.fail("wrong number of non-critical extensions in OCSP request.");
        }
        if (!((extObj = X509ExtensionUtil.fromExtensionValue(extValue = req.getExtensionValue(OCSPObjectIdentifiers.id_pkix_ocsp_nonce.getId()))) instanceof ASN1OctetString)) {
            this.fail("wrong extension type found.");
        }
        if (!this.areEqual(((ASN1OctetString)extObj).getOctets(), sampleNonce)) {
            this.fail("wrong extension value found.");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        BasicOCSPRespGenerator respGen = new BasicOCSPRespGenerator(signKP.getPublic());
        respGen.addResponse(id, CertificateStatus.GOOD);
        BasicOCSPResp resp = respGen.generate("SHA1withECDSA", signKP.getPrivate(), chain, new Date(), "BC");
    }

    private void testRSA() throws Exception {
        byte[] extValue;
        ASN1Object extObj;
        Set extOids;
        byte[] reqEnc;
        OCSPReq newReq;
        Req[] requests;
        X509Certificate[] certs;
        String signDN = "O=Bouncy Castle, C=AU";
        KeyPair signKP = OCSPTestUtil.makeKeyPair();
        X509Certificate testCert = OCSPTestUtil.makeCertificate(signKP, signDN, signKP, signDN);
        String origDN = "CN=Eric H. Echidna, E=eric@bouncycastle.org, O=Bouncy Castle, C=AU";
        GeneralName origName = new GeneralName(new X509Name(origDN));
        CertificateID id = new CertificateID("1.3.14.3.2.26", testCert, BigInteger.valueOf(1L));
        OCSPReqGenerator gen = new OCSPReqGenerator();
        gen.addRequest(new CertificateID("1.3.14.3.2.26", testCert, BigInteger.valueOf(1L)));
        OCSPReq req = gen.generate();
        if (req.isSigned()) {
            this.fail("signed but shouldn't be");
        }
        if ((certs = req.getCerts("BC")) != null) {
            this.fail("null certs expected, but not found");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        X509Certificate[] chain = new X509Certificate[1];
        gen = new OCSPReqGenerator();
        gen.setRequestorName(new GeneralName(4, new X509Principal("CN=fred")));
        gen.addRequest(new CertificateID("1.3.14.3.2.26", testCert, BigInteger.valueOf(1L)));
        chain[0] = testCert;
        req = gen.generate("SHA1withRSA", signKP.getPrivate(), chain, "BC");
        if (!req.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!req.verify(signKP.getPublic(), "BC")) {
            this.fail("signature failed to verify");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        if ((certs = req.getCerts("BC")) == null) {
            this.fail("null certs found");
        }
        if (certs.length != 1 || !certs[0].equals(testCert)) {
            this.fail("incorrect certs found in request");
        }
        if (!(newReq = new OCSPReq(reqEnc = req.getEncoded())).verify(signKP.getPublic(), "BC")) {
            this.fail("newReq signature failed to verify");
        }
        chain = new X509Certificate[1];
        gen = new OCSPReqGenerator();
        Vector<DERObjectIdentifier> oids = new Vector<DERObjectIdentifier>();
        Vector<X509Extension> values = new Vector<X509Extension>();
        byte[] sampleNonce = new byte[16];
        Random rand = new Random();
        rand.nextBytes(sampleNonce);
        gen.setRequestorName(new GeneralName(4, new X509Principal("CN=fred")));
        oids.addElement(OCSPObjectIdentifiers.id_pkix_ocsp_nonce);
        values.addElement(new X509Extension(false, (ASN1OctetString)new DEROctetString(new DEROctetString(sampleNonce))));
        gen.setRequestExtensions(new X509Extensions(oids, values));
        gen.addRequest(new CertificateID("1.3.14.3.2.26", testCert, BigInteger.valueOf(1L)));
        chain[0] = testCert;
        req = gen.generate("SHA1withRSA", signKP.getPrivate(), chain, "BC");
        if (!req.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!req.verify(signKP.getPublic(), "BC")) {
            this.fail("signature failed to verify");
        }
        if ((extOids = req.getCriticalExtensionOIDs()).size() != 0) {
            this.fail("wrong number of critical extensions in OCSP request.");
        }
        if ((extOids = req.getNonCriticalExtensionOIDs()).size() != 1) {
            this.fail("wrong number of non-critical extensions in OCSP request.");
        }
        if (!((extObj = X509ExtensionUtil.fromExtensionValue(extValue = req.getExtensionValue(OCSPObjectIdentifiers.id_pkix_ocsp_nonce.getId()))) instanceof ASN1OctetString)) {
            this.fail("wrong extension type found.");
        }
        if (!this.areEqual(((ASN1OctetString)extObj).getOctets(), sampleNonce)) {
            this.fail("wrong extension value found.");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        BasicOCSPRespGenerator respGen = new BasicOCSPRespGenerator(signKP.getPublic());
        respGen.addResponse(id, CertificateStatus.GOOD);
        BasicOCSPResp resp = respGen.generate("SHA1withRSA", signKP.getPrivate(), chain, new Date(), "BC");
        OCSPRespGenerator rGen = new OCSPRespGenerator();
        byte[] enc = rGen.generate(0, resp).getEncoded();
    }

    private void testIrregularVersionReq() throws Exception {
        OCSPReq ocspRequest = new OCSPReq(irregReq);
        X509Certificate cert = ocspRequest.getCerts("BC")[0];
        if (!ocspRequest.verify(cert.getPublicKey(), "BC")) {
            this.fail("extra version encoding test failed");
        }
    }

    public void performTest() throws Exception {
        BasicOCSPResp brep;
        OCSPResp response;
        byte[] extValue;
        ASN1Object extObj;
        Set extOids;
        byte[] reqEnc;
        OCSPReq newReq;
        Req[] requests;
        X509Certificate[] certs;
        String signDN = "O=Bouncy Castle, C=AU";
        KeyPair signKP = OCSPTestUtil.makeKeyPair();
        X509Certificate testCert = OCSPTestUtil.makeCertificate(signKP, signDN, signKP, signDN);
        String origDN = "CN=Eric H. Echidna, E=eric@bouncycastle.org, O=Bouncy Castle, C=AU";
        GeneralName origName = new GeneralName(new X509Name(origDN));
        CertificateID id = new CertificateID("1.3.14.3.2.26", testCert, BigInteger.valueOf(1L));
        id = new CertificateID("1.3.14.3.2.26", testCert, BigInteger.valueOf(1L), null);
        OCSPReqGenerator gen = new OCSPReqGenerator();
        gen.addRequest(new CertificateID("1.3.14.3.2.26", testCert, BigInteger.valueOf(1L)));
        OCSPReq req = gen.generate();
        if (req.isSigned()) {
            this.fail("signed but shouldn't be");
        }
        if ((certs = req.getCerts("BC")) != null) {
            this.fail("null certs expected, but not found");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        X509Certificate[] chain = new X509Certificate[1];
        gen = new OCSPReqGenerator();
        gen.setRequestorName(new GeneralName(4, new X509Principal("CN=fred")));
        gen.addRequest(new CertificateID("1.3.14.3.2.26", testCert, BigInteger.valueOf(1L)));
        chain[0] = testCert;
        req = gen.generate("SHA1withRSA", signKP.getPrivate(), chain, "BC");
        if (!req.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!req.verify(signKP.getPublic(), "BC")) {
            this.fail("signature failed to verify");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        if ((certs = req.getCerts("BC")) == null) {
            this.fail("null certs found");
        }
        if (certs.length != 1 || !certs[0].equals(testCert)) {
            this.fail("incorrect certs found in request");
        }
        if (!(newReq = new OCSPReq(reqEnc = req.getEncoded())).verify(signKP.getPublic(), "BC")) {
            this.fail("newReq signature failed to verify");
        }
        chain = new X509Certificate[1];
        gen = new OCSPReqGenerator();
        Vector<DERObjectIdentifier> oids = new Vector<DERObjectIdentifier>();
        Vector<X509Extension> values = new Vector<X509Extension>();
        byte[] sampleNonce = new byte[16];
        Random rand = new Random();
        rand.nextBytes(sampleNonce);
        gen.setRequestorName(new GeneralName(4, new X509Principal("CN=fred")));
        oids.addElement(OCSPObjectIdentifiers.id_pkix_ocsp_nonce);
        values.addElement(new X509Extension(false, (ASN1OctetString)new DEROctetString(new DEROctetString(sampleNonce))));
        gen.setRequestExtensions(new X509Extensions(oids, values));
        gen.addRequest(new CertificateID("1.3.14.3.2.26", testCert, BigInteger.valueOf(1L)));
        chain[0] = testCert;
        req = gen.generate("SHA1withRSA", signKP.getPrivate(), chain, "BC");
        if (!req.isSigned()) {
            this.fail("not signed but should be");
        }
        if (!req.verify(signKP.getPublic(), "BC")) {
            this.fail("signature failed to verify");
        }
        if ((extOids = req.getCriticalExtensionOIDs()).size() != 0) {
            this.fail("wrong number of critical extensions in OCSP request.");
        }
        if ((extOids = req.getNonCriticalExtensionOIDs()).size() != 1) {
            this.fail("wrong number of non-critical extensions in OCSP request.");
        }
        if (!((extObj = X509ExtensionUtil.fromExtensionValue(extValue = req.getExtensionValue(OCSPObjectIdentifiers.id_pkix_ocsp_nonce.getId()))) instanceof ASN1OctetString)) {
            this.fail("wrong extension type found.");
        }
        if (!this.areEqual(((ASN1OctetString)extObj).getOctets(), sampleNonce)) {
            this.fail("wrong extension value found.");
        }
        if (!(requests = req.getRequestList())[0].getCertID().equals(id)) {
            this.fail("Failed isFor test");
        }
        if ((response = new OCSPResp(new ByteArrayInputStream(this.testResp1))).getStatus() != 0) {
            this.fail("response status not zero.");
        }
        if (!(brep = (BasicOCSPResp)response.getResponseObject()).verify((chain = brep.getCerts("BC"))[0].getPublicKey(), "BC")) {
            this.fail("response 1 failed to verify.");
        }
        SingleResp[] singleResp = brep.getResponses();
        response = new OCSPResp(new ByteArrayInputStream(this.testResp2));
        if (response.getStatus() != 0) {
            this.fail("response status not zero.");
        }
        if (!(brep = (BasicOCSPResp)response.getResponseObject()).verify((chain = brep.getCerts("BC"))[0].getPublicKey(), "BC")) {
            this.fail("response 2 failed to verify.");
        }
        singleResp = brep.getResponses();
        OCSPRespGenerator respGen = new OCSPRespGenerator();
        OCSPResp resp = respGen.generate(0, response.getResponseObject());
        if (!resp.getResponseObject().equals(response.getResponseObject())) {
            this.fail("response fails to match");
        }
        this.testECDSA();
        this.testRSA();
        this.testIrregularVersionReq();
    }

    public static void main(String[] args) {
        Security.addProvider(new SwxaProvider());
        OCSPTest.runTest(new OCSPTest());
    }
}

